
//No part of this file can be copied or released without the consent of 
//Avalanche Technology
//										
///////////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////////
//										
//	Avalanche Technology Inc., Proprietary and Confidential	   *
//										
// 	Release:  1.0    Date 4/11/2022  	
//										
////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////
//  PART DESCRIPTION:
//
//  Technology: 22nm pMTJ STT-MRAM
//  Part:       AS302G208
//
//  Description: 2 Gigabit HIGH PERFORMANCE SERIAL PERSISTANT SRAM MEMORY
//
////////////////////////////////////////////////////////////////////////////////////
//  FILE CONTENTS : SPI/DPI/QPI RELATED TASKS 
//
////////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////////
// MODULE DECLARATION                                                             //
////////////////////////////////////////////////////////////////////////////////////

`timescale 100ps / 10ps

/////////////////////////////////////////////////////////////////
//   Datasheet Timing
/////////////////////////////////////////////////////////////////
specify 
  specparam tPOR_time  = 1; 
  specparam tCSS       = 50; 
  specparam tCSH       = 40; 
//`ifdef DDR_TIMING
//  specparam tCH        = 92.5; 
//  specparam tCL        = 92.5; 
//`else
//  specparam tCH        = 50; 
//  specparam tCL        = 50; 
//`endif
  specparam tCS        = 1300; 
  specparam tSU        = 40;
  specparam tHD        = 40;
  specparam tCO        = 90;
endspecify

/////////////////////////////////////////////////////////////////
// Select chip
/////////////////////////////////////////////////////////////////
task DL_SelectChip;
input [1:0] sel_lane;
begin
  if (sel_lane[0])
	  L0_CSn = 0;
  if (sel_lane[1])
    L1_CSn = 0;
	#(tCSS);
end
endtask
/////////////////////////////////////////////////////////////////
// Deselect chip
/////////////////////////////////////////////////////////////////
task DL_DeSelChip;
begin
	#(tCSH);
  L0_CSn = 1;
  L1_CSn = 1; 
	#(tCS);
end
endtask
/////////////////////////////////////////////////////////////////
// Send byte
////////////////////////////////////////////////////////////////
task DL_SendByte;
input [15:0] data;
input [1:0] sel_lane;
integer i;
begin
  CLK = 0;

`ifdef SUHDCHK_ZHD
  for(i=0; i<8; i=i+1)
  begin
    L0_tx[0] = sel_lane[0];
    L1_tx[0] = sel_lane[1];
    L0_SI_IO0_reg = data[7-i];
    L1_SI_IO0_reg = data[15-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    L0_tx[0] = 0;
    L1_tx[0] = 0;
    L0_SI_IO0_reg = 0;
    L1_SI_IO0_reg = 0;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end

`elsif SUHDCHK_CKFE
  for(i=0; i<8; i=i+1)
  begin
    CLK = 0;
    L0_tx[0] = sel_lane[0];
    L1_tx[0] = sel_lane[1];
    L0_SI_IO0_reg = data[7-i];
    L1_SI_IO0_reg = data[15-i];
    #(tCL);
    CLK = 1;
    #(tCH)
    L0_tx[0] = 0;
    L1_tx[0] = 0;
    CLK = 0;
  end

`else
  L0_tx[0] = sel_lane[0];
  L1_tx[0] = sel_lane[1];
  for(i=0; i<8; i=i+1)
  begin
    L0_SI_IO0_reg = data[7-i];
    L1_SI_IO0_reg = data[15-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    L0_SI_IO0_reg = 0;
    L1_SI_IO0_reg = 0;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end
  tx[0] = 0;
`endif

end
endtask
/////////////////////////////////////////////////////////////////
// Send Quad byte
/////////////////////////////////////////////////////////////////
task DL_SendByteQuad;
input [15:0] data;
input [1:0] sel_lane;
integer i;
begin
  CLK = 0;
  
`ifdef SUHDCHK_ZHD
  for(i=0; i<8; i=i+4)
  begin
    L0_tx = (sel_lane[0])? 4'b1111 : 0;
    L1_tx = (sel_lane[1])? 4'b1111 : 0;
    L0_IO3_reg = data[7-i];
    L0_WPn_IO2_reg = data[7-1-i];
    L0_SO_IO1_reg = data[7-2-i];
    L0_SI_IO0_reg = data[7-3-i];
    L1_IO3_reg = data[15-i];
    L1_WPn_IO2_reg = data[15-1-i];
    L1_SO_IO1_reg = data[15-2-i];
    L1_SI_IO0_reg = data[15-3-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    L0_tx = 4'b0000;
    L1_tx = 4'b0000;
    L0_IO3_reg = 0; 
    L0_WPn_IO2_reg = 0;
    L0_SO_IO1_reg = 0;
    L0_SI_IO0_reg = 0;
    L1_IO3_reg = 0; 
    L1_WPn_IO2_reg = 0;
    L1_SO_IO1_reg = 0;
    L1_SI_IO0_reg = 0;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end

`elsif SUHDCHK_CKFE
  for(i=0; i<8; i=i+4)
  begin
    CLK = 0;
    L0_tx = (sel_lane[0])? 4'b1111 : 0;
    L1_tx = (sel_lane[1])? 4'b1111 : 0;
    L0_IO3_reg = data[7-i];
    L0_WPn_IO2_reg = data[7-1-i];
    L0_SO_IO1_reg = data[7-2-i];
    L0_SI_IO0_reg = data[7-3-i];
    L1_IO3_reg = data[15-i];
    L1_WPn_IO2_reg = data[15-1-i];
    L1_SO_IO1_reg = data[15-2-i];
    L1_SI_IO0_reg = data[15-3-i];
    #(tCL);
    CLK = 1;
    #(tCH)
    L0_tx = 4'b0000;
    L1_tx = 4'b0000;
    CLK = 0;
  end

`else
  L0_tx = (sel_lane[0])? 4'b1111 : 0;
  L1_tx = (sel_lane[1])? 4'b1111 : 0;
  for(i=0; i<8; i=i+4)
  begin
    L0_IO3_reg = data[7-i];
    L0_WPn_IO2_reg = data[7-1-i];
    L0_SO_IO1_reg = data[7-2-i];
    L0_SI_IO0_reg = data[7-3-i];
    L1_IO3_reg = data[15-i];
    L1_WPn_IO2_reg = data[15-1-i];
    L1_SO_IO1_reg = data[15-2-i];
    L1_SI_IO0_reg = data[15-3-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    L0_IO3_reg = 0; 
    L0_WPn_IO2_reg = 0;
    L0_SO_IO1_reg = 0;
    L0_SI_IO0_reg = 0;
    L1_IO3_reg = 0; 
    L1_WPn_IO2_reg = 0;
    L1_SO_IO1_reg = 0;
    L1_SI_IO0_reg = 0;
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end
  L0_tx = 4'b0000;
  L1_tx = 4'b0000;
`endif
end
endtask
/////////////////////////////////////////////////////////////////
// Send byte DDR
////////////////////////////////////////////////////////////////
task DL_SendByte_DDR;
input [15:0] data;
input [1:0] sel_lane;
integer i;
begin
  CLK = 0;

`ifdef SUHDCHK_ZHD
  for(i=0; i<8; i=i+2)
  begin
    L0_tx[0] = sel_lane[0];
    L1_tx[0] = sel_lane[1];
    L0_SI_IO0_reg = data[7-i];
    L1_SI_IO0_reg = data[15-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    L0_tx[0] = 0;
    L1_tx[0] = 0;
    L0_SI_IO0_reg = 0;
    L1_SI_IO0_reg = 0;
    #(tCH-tHD-tSU);
    L0_tx[0] = sel_lane[0];
    L1_tx[0] = sel_lane[1];
    L0_SI_IO0_reg = data[6-i];
    L1_SI_IO0_reg = data[14-i];
    #(tSU);
    CLK = 0;
    #(tHD)
    L0_tx[0] = 0;
    L1_tx[0] = 0;
    L0_SI_IO0_reg = 0;
    L1_SI_IO0_reg = 0;
    #(tCL-tSU-tHD);
  end

`elsif SUHDCHK_CKFE
  L0_tx[0] = sel_lane[0];
  L1_tx[0] = sel_lane[1];
  for(i=0; i<8; i=i+2)
  begin
    CLK = 0;
    #(tCL/2.0)
    L0_SI_IO0_reg = data[7-i];
    L1_SI_IO0_reg = data[15-i];
    #(tCL/2.0);
    CLK = 1;
    #(tCH/2.0)
    L0_SI_IO0_reg = data[6-i];
    L1_SI_IO0_reg = data[14-i];
    #(tCH/2.0)
    CLK = 0;
  end
  L0_tx[0] = 0;
  L1_tx[0] = 0;

`else
  L0_tx[0] = sel_lane[0];
  L1_tx[0] = sel_lane[1];
  for(i=0; i<8; i=i+2)
  begin
    L0_SI_IO0_reg = data[7-i];
    L1_SI_IO0_reg = data[15-i];
    #(tSU);
    CLK = 1;
    #(tHD)
    L0_SI_IO0_reg = data[6-i];
    L1_SI_IO0_reg = data[14-i];
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
  end
  L0_tx[0] = 0;
  L1_tx[0] = 0;
`endif

end
endtask
/////////////////////////////////////////////////////////////////
// Send Quad byte DDR
/////////////////////////////////////////////////////////////////
task DL_SendByteQuad_DDR;
input [15:0] data;
input [1:0] sel_lane;
integer i;
begin
  CLK = 0;
  
`ifdef SUHDCHK_ZHD
    L0_tx = (sel_lane[0])? 4'b1111 : 0;
    L1_tx = (sel_lane[1])? 4'b1111 : 0;
    L0_IO3_reg = data[7];
    L0_WPn_IO2_reg = data[6];
    L0_SO_IO1_reg = data[5];
    L0_SI_IO0_reg = data[4];
    L1_IO3_reg = data[15];
    L1_WPn_IO2_reg = data[14];
    L1_SO_IO1_reg = data[13];
    L1_SI_IO0_reg = data[12];
    #(tSU);
    CLK = 1;
    #(tHD)
    L0_tx = 4'b0000;
    L1_tx = 4'b0000;
    L0_IO3_reg = 0; 
    L0_WPn_IO2_reg = 0;
    L0_SO_IO1_reg = 0;
    L0_SI_IO0_reg = 0;
    L1_IO3_reg = 0; 
    L1_WPn_IO2_reg = 0;
    L1_SO_IO1_reg = 0;
    L1_SI_IO0_reg = 0;
    #(tCH-tHD-tSU);
    L0_tx = (sel_lane[0])? 4'b1111 : 0;
    L1_tx = (sel_lane[1])? 4'b1111 : 0;
    L0_IO3_reg = data[3];
    L0_WPn_IO2_reg = data[2];
    L0_SO_IO1_reg = data[1];
    L0_SI_IO0_reg = data[0];
    L1_IO3_reg = data[11];
    L1_WPn_IO2_reg = data[10];
    L1_SO_IO1_reg = data[9];
    L1_SI_IO0_reg = data[8];
    #(tSU);
    CLK = 0;
    #(tHD)
    L0_tx = 4'b0000;
    L1_tx = 4'b0000;
    L0_IO3_reg = 0; 
    L0_WPn_IO2_reg = 0;
    L0_SO_IO1_reg = 0;
    L0_SI_IO0_reg = 0;
    L1_IO3_reg = 0; 
    L1_WPn_IO2_reg = 0;
    L1_SO_IO1_reg = 0;
    L1_SI_IO0_reg = 0;
    #(tCL-tSU-tHD);

`elsif SUHDCHK_CKFE
    L0_tx = (sel_lane[0])? 4'b1111 : 0;
    L1_tx = (sel_lane[1])? 4'b1111 : 0;
    CLK = 0;
    #(tCL/2.0)
    L0_IO3_reg = data[7];
    L0_WPn_IO2_reg = data[6];
    L0_SO_IO1_reg = data[5];
    L0_SI_IO0_reg = data[4];
    L1_IO3_reg = data[15];
    L1_WPn_IO2_reg = data[14];
    L1_SO_IO1_reg = data[13];
    L1_SI_IO0_reg = data[12];
    #(tCL/2.0);
    CLK = 1;
    #(tCH/2.0)
    L0_IO3_reg = data[3];
    L0_WPn_IO2_reg = data[2];
    L0_SO_IO1_reg = data[1];
    L0_SI_IO0_reg = data[0];
    L1_IO3_reg = data[11];
    L1_WPn_IO2_reg = data[10];
    L1_SO_IO1_reg = data[9];
    L1_SI_IO0_reg = data[8];
    #(tCH/2.0);
    CLK = 0;
    L0_tx = 4'b0000;
    L1_tx = 4'b0000;

`else
    L0_tx = (sel_lane[0])? 4'b1111 : 0;
    L1_tx = (sel_lane[1])? 4'b1111 : 0;
    L0_IO3_reg = data[7];
    L0_WPn_IO2_reg = data[6];
    L0_SO_IO1_reg = data[5];
    L0_SI_IO0_reg = data[4];
    L1_IO3_reg = data[15];
    L1_WPn_IO2_reg = data[14];
    L1_SO_IO1_reg = data[13];
    L1_SI_IO0_reg = data[12];
    #(tSU);
    CLK = 1;
    #(tHD)
    L0_IO3_reg = data[3];
    L0_WPn_IO2_reg = data[2];
    L0_SO_IO1_reg = data[1];
    L0_SI_IO0_reg = data[0];
    L1_IO3_reg = data[11];
    L1_WPn_IO2_reg = data[10];
    L1_SO_IO1_reg = data[9];
    L1_SI_IO0_reg = data[8];
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
    L0_tx = 4'b0000;
    L1_tx = 4'b0000;
`endif
end
endtask
/////////////////////////////////////////////////////////////////
// Task to receive byte
/////////////////////////////////////////////////////////////////
task DL_RcvByte;
output [15:0] data;
integer i;
begin
  CLK = 0;
  L0_tx=0;
  L1_tx=0;
`ifdef SUHDCHK_CKFE
  for(i=0; i<8; i=i+1)
  begin
    CLK = 0;
    #(tCL)
    CLK = 1;
    #(tCH)
    CLK = 0;
    data[7-i]  = L0_SO_IO1;
    data[15-i] = L1_SO_IO1;
  end
`else
  for(i=0; i<8; i=i+1)
  begin
    //SI_IO0_reg = 0;
    #(tSU);
    CLK = 1;
    #(tHD)
    //SI_IO0_reg = 0;
    #(tCH-tHD);
    //SO_IO1_reg = 1'bz;
    //SI_IO0_reg = 0;
    data[7-i]  = L0_SO_IO1;
    data[15-i] = L1_SO_IO1;
    CLK = 0;
    #(tCL-tSU);
  end
`endif
end
endtask
/////////////////////////////////////////////////////////////////
// Task to receive Quadbyte
/////////////////////////////////////////////////////////////////
task DL_RcvByteQuad;
output [15:0] data;
integer i;
begin
  CLK = 0;
  L0_tx=0;
  L1_tx=0;
`ifdef SUHDCHK_CKFE
  for(i=0; i<8; i=i+4)
  begin
    CLK = 0;
    #(tCL)
    CLK = 1;
    #(tCH)
    CLK = 0;
    data[7-i]   = L0_IO3;
    data[7-1-i] = L0_WPn_IO2;
    data[7-2-i] = L0_SO_IO1;
    data[7-3-i] = L0_SI_IO0;
    data[15-i]   = L1_IO3;
    data[15-1-i] = L1_WPn_IO2;
    data[15-2-i] = L1_SO_IO1;
    data[15-3-i] = L1_SI_IO0;
  end
`else
  for(i=0; i<8; i=i+4)
  begin
    /*IO3_reg = 0; 
    WPn_IO2_reg = 0;
    SO_IO1_reg = 0; 
    SI_IO0_reg = 0; */
    #(tSU);
    CLK = 1;
    #(tHD)
    /*IO3_reg = 0; 
    WPn_IO2_reg = 0;
    SO_IO1_reg = 0;
    SI_IO0_reg = 0;*/
    #(tCH-tHD);
    /*IO3_reg=1'bz;
    WPn_IO2_reg=1'bz;
    SO_IO1_reg=1'bz;
    SI_IO0_reg=1'bz;*/
    data[7-i]   = L0_IO3;
    data[7-1-i] = L0_WPn_IO2;
    data[7-2-i] = L0_SO_IO1;
    data[7-3-i] = L0_SI_IO0;
    data[15-i]   = L1_IO3;
    data[15-1-i] = L1_WPn_IO2;
    data[15-2-i] = L1_SO_IO1;
    data[15-3-i] = L1_SI_IO0;
    CLK = 0;
    #(tCL-tSU);
  end
`endif
end
endtask
/////////////////////////////////////////////////////////////////
// Task to receive byte DDR
/////////////////////////////////////////////////////////////////
task DL_RcvByte_DDR;
output [15:0] data;
integer i;
begin
  CLK = 0;
  L0_tx = 0;
  L1_tx = 0;
`ifdef SUHDCHK_CKFE
  for(i=0; i<8; i=i+2)
  begin
    CLK = 0;
    #(tCL)
    CLK = 1;
    data[7-i] = L0_SO_IO1;
    data[15-i] = L1_SO_IO1;
    #(tCH)
    CLK = 0;
    data[6-i] = L0_SO_IO1;
    data[14-i] = L1_SO_IO1;
  end
`else
  for(i=0; i<8; i=i+2)
  begin
    //SI_IO0_reg = 0;
    #(tSU);
    CLK = 1;
    data[7-i] = L0_SO_IO1;
    data[15-i] = L1_SO_IO1;
    #(tHD)
    //SI_IO0_reg = 0;
    #(tCH-tHD);
    //SO_IO1_reg = 1'bz;
    //SI_IO0_reg = 0;
    data[6-i] = L0_SO_IO1;
    data[14-i] = L1_SO_IO1;
    CLK = 0;
    #(tCL-tSU);
  end
`endif
end
endtask
/////////////////////////////////////////////////////////////////
// Task to receive Quadbyte DDR
/////////////////////////////////////////////////////////////////
task DL_RcvByteQuad_DDR;
output [15:0] data;
integer i;
begin
  CLK = 0;
  L0_tx = 0;
  L1_tx = 0;
`ifdef SUHDCHK_CKFE
    CLK = 0;
    #(tCL)
    CLK = 1;
    data[7]  = L0_IO3;
    data[6]  = L0_WPn_IO2;
    data[5]  = L0_SO_IO1;
    data[4]  = L0_SI_IO0;
    data[15] = L1_IO3;
    data[14] = L1_WPn_IO2;
    data[13] = L1_SO_IO1;
    data[12] = L1_SI_IO0;
    #(tCH)
    CLK = 0;
    data[3]  = L0_IO3;
    data[2]  = L0_WPn_IO2;
    data[1]  = L0_SO_IO1;
    data[0]  = L0_SI_IO0;
    data[11] = L1_IO3;
    data[10] = L1_WPn_IO2;
    data[9]  = L1_SO_IO1;
    data[8]  = L1_SI_IO0;
`else
    //IO3_reg = 0; 
    //WPn_IO2_reg = 0;
    //SO_IO1_reg = 0; 
    //SI_IO0_reg = 0; 
    #(tSU);
    CLK = 1;
    data[7]  = L0_IO3;
    data[6]  = L0_WPn_IO2;
    data[5]  = L0_SO_IO1;
    data[4]  = L0_SI_IO0;
    data[15] = L1_IO3;
    data[14] = L1_WPn_IO2;
    data[13] = L1_SO_IO1;
    data[12] = L1_SI_IO0;
    #(tHD)
    //IO3_reg = 0; 
    //WPn_IO2_reg = 0;
    //SO_IO1_reg = 0;
    //SI_IO0_reg = 0;
    #(tCH-tHD);
    //IO3_reg=1'bz;
    //WPn_IO2_reg=1'bz;
    //SO_IO1_reg=1'bz;
    //SI_IO0_reg=1'bz;
    data[3]  = L0_IO3;
    data[2]  = L0_WPn_IO2;
    data[1]  = L0_SO_IO1;
    data[0]  = L0_SI_IO0;
    data[11] = L1_IO3;
    data[10] = L1_WPn_IO2;
    data[9]  = L1_SO_IO1;
    data[8]  = L1_SI_IO0;
    CLK = 0;
  end
`endif
end
endtask
/////////////////////////////////////////////////////////////////
// Send command
/////////////////////////////////////////////////////////////////
task DL_Cmd;
input [2:0] Mode;
input [7:0] L0_cmd, L1_cmd;
input [1:0] sel_lane;
begin
	if (Mode == 3'h1)
	begin
	DL_SendByte({L1_cmd, L0_cmd}, sel_lane);
	end
  else if (Mode == 3'h4)
	begin
	DL_SendByteQuad({L1_cmd, L0_cmd}, sel_lane);
	end
end
endtask
/////////////////////////////////////////////////////////////////
// Send 32-bit address
/////////////////////////////////////////////////////////////////
task DL_Addr_32bit;
input [2:0] Mode;
input [31:0] L0_addr, L1_addr;
input [1:0] sel_lane;
begin
	if (Mode == 3'h1)
	begin
	DL_SendByte({L1_addr[31:24], L0_addr[31:24]}, sel_lane);
	DL_SendByte({L1_addr[23:16], L0_addr[23:16]}, sel_lane);
	DL_SendByte({L1_addr[15:8],  L0_addr[15:8]},  sel_lane);
	DL_SendByte({L1_addr[7:0],   L0_addr[7:0]},   sel_lane);
	end

  else if (Mode == 3'h4)
	begin
	DL_SendByteQuad({L1_addr[31:24], L0_addr[31:24]}, sel_lane);
	DL_SendByteQuad({L1_addr[23:16], L0_addr[23:16]}, sel_lane);
	DL_SendByteQuad({L1_addr[15:8],  L0_addr[15:8]},  sel_lane);
	DL_SendByteQuad({L1_addr[7:0],   L0_addr[7:0]},   sel_lane);
	end
end
endtask
/////////////////////////////////////////////////////////////////
// Send 32-bit address DDR
/////////////////////////////////////////////////////////////////
task DL_Addr_32bit_DDR;
input [2:0] Mode;
input [31:0] L0_addr, L1_addr;
input [1:0] sel_lane;
begin
	if (Mode == 3'h1)
	begin
	DL_SendByte_DDR({L1_addr[31:24], L0_addr[31:24]}, sel_lane);
	DL_SendByte_DDR({L1_addr[23:16], L0_addr[23:16]}, sel_lane);
	DL_SendByte_DDR({L1_addr[15:8],  L0_addr[15:8]},  sel_lane);
	DL_SendByte_DDR({L1_addr[7:0],   L0_addr[7:0]},   sel_lane);
	end

  else if (Mode == 3'h4)
	begin
	DL_SendByteQuad_DDR({L1_addr[31:24], L0_addr[31:24]}, sel_lane);
	DL_SendByteQuad_DDR({L1_addr[23:16], L0_addr[23:16]}, sel_lane);
	DL_SendByteQuad_DDR({L1_addr[15:8],  L0_addr[15:8]},  sel_lane);
	DL_SendByteQuad_DDR({L1_addr[7:0],   L0_addr[7:0]},   sel_lane);
	end
end
endtask
/////////////////////////////////////////////////////////////////
// Send 24-bit address
/////////////////////////////////////////////////////////////////
task DL_Addr_24bit;
input [2:0] Mode;
input [23:0] L0_addr, L1_addr;
input [1:0] sel_lane;
begin
	if (Mode == 3'h1)
	begin
	DL_SendByte({L1_addr[23:16], L0_addr[23:16]}, sel_lane);
	DL_SendByte({L1_addr[15:8],  L0_addr[15:8]},  sel_lane);
	DL_SendByte({L1_addr[7:0],   L0_addr[7:0]},   sel_lane);
	end

  else if (Mode == 3'h4)
	begin
	DL_SendByteQuad({L1_addr[23:16], L0_addr[23:16]}, sel_lane);
	DL_SendByteQuad({L1_addr[15:8],  L0_addr[15:8]},  sel_lane);
	DL_SendByteQuad({L1_addr[7:0],   L0_addr[7:0]},   sel_lane);
	end
end
endtask
/////////////////////////////////////////////////////////////////
// Send 24-bit address DDR
/////////////////////////////////////////////////////////////////
task DL_Addr_24bit_DDR;
input [2:0] Mode;
input [23:0] L0_addr, L1_addr;
input [1:0] sel_lane;
begin
	if (Mode == 3'h1)
	begin
	DL_SendByte_DDR({L1_addr[23:16], L0_addr[23:16]}, sel_lane);
	DL_SendByte_DDR({L1_addr[15:8],  L0_addr[15:8]},  sel_lane);
	DL_SendByte_DDR({L1_addr[7:0],   L0_addr[7:0]},   sel_lane);
	end

  else if (Mode == 3'h4)
	begin
	DL_SendByteQuad_DDR({L1_addr[23:16], L0_addr[23:16]}, sel_lane);
	DL_SendByteQuad_DDR({L1_addr[15:8],  L0_addr[15:8]},  sel_lane);
	DL_SendByteQuad_DDR({L1_addr[7:0],   L0_addr[7:0]},   sel_lane);
	end
end
endtask
/////////////////////////////////////////////////////////////////
// Send XIP mode byte
/////////////////////////////////////////////////////////////////
task DL_ModeByte;
input [2:0] Mode;
input [7:0] L0_data, L1_data;
input [1:0] sel_lane;
begin
	if (Mode == 3'h1)
	begin
	DL_SendByte({L1_data, L0_data}, sel_lane);
	end
  else if (Mode == 3'h4)
	begin
	DL_SendByteQuad({L1_data, L0_data}, sel_lane);
	end
end
endtask
/////////////////////////////////////////////////////////////////
// Send XIP mode byte DDR
/////////////////////////////////////////////////////////////////
task DL_ModeByte_DDR;
input [2:0] Mode;
input [7:0] L0_data, L1_data;
input [1:0] sel_lane;
begin
	if (Mode == 3'h1)
	begin
	DL_SendByte_DDR({L1_data, L0_data}, sel_lane);
	end
  else if (Mode == 3'h4)
	begin
	DL_SendByteQuad_DDR({L1_data, L0_data}, sel_lane);
	end
end
endtask
/////////////////////////////////////////////////////////////////
// Task to send latency
/////////////////////////////////////////////////////////////////
task DL_Latency;
input [7:0] num;
integer i;
begin
  CLK = 0;
  L0_tx = 0;
  L1_tx = 0;
  for(i=0; i<num; i=i+1)
  begin
`ifdef SUHDCHK_CKFE
    CLK = 0;
    #(tCL);
    CLK = 1;
    #(tCH);
    CLK = 0;
`else
    #(tSU);
    CLK = 1;
    #(tHD)
    #(tCH-tHD);
    CLK = 0;
    #(tCL-tSU);
`endif
  end
end
endtask
/////////////////////////////////////////////////////////////////
// SPI enable for Modes
/////////////////////////////////////////////////////////////////
task DL_SPIEnable;
input [2:0] Mode;
input [1:0] sel_lane;
begin
  DL_SelectChip(sel_lane);
  DL_Cmd(Mode, 8'hff, 8'hff, sel_lane);
  DL_DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// QPI enable for Modes
/////////////////////////////////////////////////////////////////
task DL_QPIEnable;
input [2:0] Mode;
input [1:0] sel_lane;
begin
  DL_SelectChip(sel_lane);
  DL_Cmd(Mode, 8'h38, 8'h38, sel_lane);
  DL_DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// Write enable
/////////////////////////////////////////////////////////////////
task DL_WriteEnable;
input [2:0] Mode;
input [1:0] sel_lane;
begin
 	DL_SelectChip(sel_lane);
 	DL_Cmd(Mode, 8'h06, 8'h06, sel_lane);
 	DL_DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// Write disable
/////////////////////////////////////////////////////////////////
task DL_WriteDisable;
input [2:0] Mode;
input [1:0] sel_lane;
begin
	DL_SelectChip(sel_lane);
 	DL_Cmd(Mode, 8'h04, 8'h04, sel_lane);
 	DL_DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// Write Any Register
/////////////////////////////////////////////////////////////////
task DL_WriteReg;
input [2:0] Mode;
input [31:0] addr; // identical for both lanes
input [31:0] data; // identical for both lanes
input [1:0] sel_lane;

begin
  $display("[INFO] Write Any Register : Address %x data %x", addr, data);
  if (sel_lane[0])
    $display("       -> Writing to lane 0");
  if (sel_lane[1])
    $display("       -> Writing to lane 1");
  case(addr)
    32'h0: $display("       --> Status Register");
    32'h1: $display("       --> Interrupt Status Register");
    32'h2: $display("       --> Config Register 1");
    32'h3: $display("       --> Config Register 2");
    32'h4: $display("       --> Interrupt Config Register");
    32'h5: $display("       --> ECC Test - Data Input Register");
    32'h6: $display("       --> ECC Test - Error Injection Register");
    32'h7: $display("       --> ECC Test - Data Output Register");
    32'h8: $display("       --> ECC Test - Error Count Register");
    32'h30: $display("       --> Device Identification Register");
    32'h40: $display("       --> Device Unique Identification Register");
    32'h50: $display("       --> Serial Number Register");
    default: $display("       --> Invalid Register address");
  endcase

  DL_SelectChip(sel_lane);
  DL_Cmd(Mode, 8'h71, 8'h71, sel_lane);
  DL_Addr_32bit(Mode, addr, addr, sel_lane);

  if (Mode == 3'h1)
  begin
    case(addr)
      32'h5, 32'h6: begin
        DL_SendByte({data[31:24], data[31:24]}, sel_lane);
        DL_SendByte({data[23:16], data[23:16]}, sel_lane);
        DL_SendByte({data[15:8],  data[15:8]},  sel_lane);
        DL_SendByte({data[7:0],   data[7:0]},   sel_lane);
      end
      default: begin
        DL_SendByte({data[7:0], data[7:0]}, sel_lane);
      end
    endcase
  end

  DL_DeSelChip();
end
endtask

/////////////////////////////////////////////////////////////////
// Read Any Register
/////////////////////////////////////////////////////////////////
task DL_ReadReg;
input [2:0] Mode;
input [31:0] addr; // identical addr for both lanes
input [7:0] data; // will compare the same data
input [7:0] latency_val;
input [1:0] sel_lane;
reg [15:0] data_rcv;
reg error_flag;
integer i;
reg [31:0] L0_id, L1_id;
reg [63:0] L0_uid, L1_uid;
begin

  $display("[INFO] Read Any Register : Address %x data %x", addr, data);
  case(addr)
    32'h0: $display("       --> Status Register");
    32'h1: $display("       --> Interrupt Status Register");
    32'h2: $display("       --> Config Register 1");
    32'h3: $display("       --> Config Register 2");
    32'h4: $display("       --> Interrupt Config Register");
    32'h5: $display("       --> ECC Test - Data Input Register");
    32'h6: $display("       --> ECC Test - Error Injection Register");
    32'h7: $display("       --> ECC Test - Data Output Register");
    32'h8: $display("       --> ECC Test - Error Count Register");
    32'h30: $display("       --> Device Identification Register");
    32'h40: $display("       --> Device Unique Identification Register");
    32'h50: $display("       --> Serial Number Register");
    default: $display("       --> Invalid Register address");
  endcase

  DL_SelectChip(sel_lane);
  DL_Cmd(Mode, 8'h65, 8'h65, sel_lane);
  DL_Addr_32bit(Mode, addr, addr, sel_lane);

  case(addr)
    // 32-bit output
    32'h5, 32'h6, 32'h7, 32'h8, 32'h30:
    begin
      DL_Latency(latency_val);
      for (i=0; i<4; i=i+1) begin
        DL_RcvByte(data_rcv);
        L0_id = {L0_id[23:0], data_rcv[7:0]};
        L1_id = {L1_id[23:0], data_rcv[15:8]};
      end
    end

    // 64-bit output 
    32'h40, 32'h50:
    begin
      DL_Latency(latency_val);
      for (i=0; i<8; i=i+1) begin
        DL_RcvByte(data_rcv);
        L0_uid = {L0_uid[55:0], data_rcv[7:0]};
        L1_uid = {L1_uid[55:0], data_rcv[15:8]};
      end
    end

    // 8-bit output 
    default: begin
      DL_Latency(latency_val);
      DL_RcvByte(data_rcv);
    end
  endcase

  DL_DeSelChip();

  error_flag = 0;
  case(addr)
    32'h0:
      if (data_rcv !== ({data, data} & 16'hfcfc))
        error_flag = 1'b1;
    32'h2: //CR1
      if ((data_rcv & 16'hE7E7) !== ({data, data} & 16'hE7E7))
        error_flag = 1'b1;
    32'h3: //CR2
      if ((data_rcv & 16'h0F0F) !== ({data, data} & 16'h0F0F))
        error_flag = 1'b1;
    32'h4: //INTCR
      if ((data_rcv & 16'h6F6F) !== ({data, data} & 16'h6F6F))
        error_flag = 1'b1;
    32'h1: //ISR
      if ((data_rcv & 16'hFFFF) !== ({data, data} & 16'hFFFF))
        error_flag = 1'b1;

  endcase

  if (error_flag) begin
    if (sel_lane[0])
      $display("[ERROR] Register Compare Error (Reg 0x%x) : Lane 0 Expected 0x%x Received 0x%x (time:%t)", addr, data, data_rcv[7:0], $time);
    if (sel_lane[1])
      $display("[ERROR] Register Compare Error (Reg 0x%x) : Lane 1 Expected 0x%x Received 0x%x (time:%t)", addr, data, data_rcv[15:8], $time);
  end
  else
    case(addr)
      32'h0, 32'h1, 32'h2, 32'h3, 32'h4: begin
        if (sel_lane[0])
          $display("[INFO] Register Compare Pass (Reg 0x%x) : Lane 0 Expected 0x%x Received 0x%x", addr, data, data_rcv[7:0]);
        if (sel_lane[1])
          $display("[INFO] Register Compare Pass (Reg 0x%x) : Lane 1 Expected 0x%x Received 0x%x", addr, data, data_rcv[15:8]);
      end
      32'h5, 32'h6, 32'h7, 32'h8: begin
        if (sel_lane[0])
          $display("[INFO] Register Read ECC register : Lane 0 0x%x", L0_id);
        if (sel_lane[1])
          $display("[INFO] Register Read ECC register : Lane 1 0x%x", L1_id);
      end
      32'h30: begin
        if (sel_lane[0])
          $display("[INFO] Register Read ID : Lane 0 0x%x", L0_id);
        if (sel_lane[1])
          $display("[INFO] Register Read ID : Lane 1 0x%x", L1_id);
      end
      32'h40: begin
        if (sel_lane[0])
          $display("[INFO] Register Read UID : Lane 0 0x%x", L0_uid);
        if (sel_lane[1])
          $display("[INFO] Register Read UID : Lane 1 0x%x", L1_uid);
      end
      32'h50: begin
        if (sel_lane[0])
          $display("[INFO] Register Read SN : Lane 0 0x%x", L0_uid);
        if (sel_lane[1])
          $display("[INFO] Register Read SN : Lane 1 0x%x", L1_uid);
      end
      default:
        $display("[ERROR] Register Compare not performed due to invalid/unsupported address (Reg 0x%x)", addr);
    endcase
end
endtask
/////////////////////////////////////////////////////////////////
// Memory Write (0x02) Sequential
/////////////////////////////////////////////////////////////////
task DL_WriteSeq;
input [2:0] Mode;
input [31:0] addr;
input [15:0] data_base;
input [7:0] data_inc; // increment for each lane
input [23:0] num_data_i;
input [1:0] sel_lane;
reg [15:0] data;
integer i; 
begin
  $display("[INFO] Mem write (sequential)  : Address %x num_data %x data_base %x data_inc %x",addr, num_data_i, data_base, data_inc);
  data = data_base;
  DL_SelectChip(sel_lane);
  DL_Cmd(Mode, 8'h02, 8'h02, sel_lane);
  DL_Addr_32bit(Mode, addr, addr, sel_lane);
  for (i=0; i<num_data_i; i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      DL_SendByte(data, sel_lane);
    end
    else if (Mode == 3'h4)
    begin
      DL_SendByteQuad(data, sel_lane);
    end
    if (sel_lane[0]) begin
      $display("[INFO] -> Writing to Lane 0 Address %x data %x", addr+i, data[7:0]);
    end
    if (sel_lane[1]) begin
      $display("[INFO] -> Writing to Lane 1 Address %x data %x", addr+i, data[15:8]);
    end
    data = {(data[15:8] + data_inc), (data[7:0] + data_inc)};
  end
  DL_DeSelChip();
end
endtask
/////////////////////////////////////////////////////////////////
// Memory Read (0x13) Sequential
/////////////////////////////////////////////////////////////////
task DL_ReadSeq;
input [2:0] Mode;
input [31:0] addr;
input [15:0] data_base;
input [7:0] data_inc; // increment for each lane
input [23:0] num_data_o;
input [1:0] sel_lane;
integer i;
reg [15:0] data_rcv;
reg [15:0] data;
begin
  $display("[INFO] Mem read (sequential) : Address %x num_data %x data_base %x data_inc %x",addr, num_data_o, data_base, data_inc);
  data = data_base;
  DL_SelectChip(sel_lane);
  DL_Cmd(Mode, 8'h13, 8'h13, sel_lane);
  DL_Addr_32bit(Mode, addr, addr, sel_lane);

  for (i=0; i<num_data_o; i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      DL_RcvByte(data_rcv);
    end
    else if (Mode == 3'h4)
    begin
      DL_RcvByteQuad(data_rcv);
    end

    /* Lane 0 */
    if (sel_lane[0]) begin
    if (data_rcv[7:0] !== data[7:0])
      $display("[ERROR] Mem Read Compare Error : Lane 0 Expected %x Received %x",
               data[7:0], data_rcv[7:0]);
    else
      $display("[INFO] Mem Read Compare Pass : Lane 0 Expected %x Received %x",
               data[7:0], data_rcv[7:0]);
    end
    /* Lane 1 */
    if (sel_lane[1]) begin
    if (data_rcv[15:8] !== data[15:8])
      $display("[ERROR] Mem Read Compare Error : Lane 1 Expected %x Received %x",
               data[15:8], data_rcv[15:8]);
    else
      $display("[INFO] Mem Read Compare Pass : Lane 1 Expected %x Received %x",
               data[15:8], data_rcv[15:8]);
    end

    data = {(data[15:8] + data_inc), (data[7:0] + data_inc)};
  end
  DL_DeSelChip();

end
endtask
/////////////////////////////////////////////////////////////////
// Memory Read (0x03) Sequential
/////////////////////////////////////////////////////////////////
task DL_ReadSeq3byte;
input [2:0] Mode;
input [31:0] addr;
input [15:0] data_base;
input [7:0] data_inc; // increment for each lane
input [23:0] num_data_o;
input [1:0] sel_lane;
integer i;
reg [15:0] data_rcv;
reg [15:0] data;
begin
  $display("[INFO] Mem read (sequential) : Address %x num_data %x data_base %x data_inc %x",addr, num_data_o, data_base, data_inc);
  data = data_base;
  DL_SelectChip(sel_lane);
  DL_Cmd(Mode, 8'h03, 8'h03, sel_lane);
  DL_Addr_24bit(Mode, addr[23:0], addr[23:0], sel_lane);

  for (i=0; i<num_data_o; i=i+1)
  begin
    if (Mode == 3'h1)
    begin
      DL_RcvByte(data_rcv);
    end
    else if (Mode == 3'h4)
    begin
      DL_RcvByteQuad(data_rcv);
    end

    /* Lane 0 */
    if (sel_lane[0]) begin
    if (data_rcv[7:0] !== data[7:0])
      $display("[ERROR] Mem Read Compare Error : Lane 0 Expected %x Received %x",
               data[7:0], data_rcv[7:0]);
    else
      $display("[INFO] Mem Read Compare Pass : Lane 0 Expected %x Received %x",
               data[7:0], data_rcv[7:0]);
    end
    /* Lane 1 */
    if (sel_lane[1]) begin
    if (data_rcv[15:8] !== data[15:8])
      $display("[ERROR] Mem Read Compare Error : Lane 1 Expected %x Received %x",
               data[15:8], data_rcv[15:8]);
    else
      $display("[INFO] Mem Read Compare Pass : Lane 1 Expected %x Received %x",
               data[15:8], data_rcv[15:8]);
    end

    data = {(data[15:8] + data_inc), (data[7:0] + data_inc)};
  end
  DL_DeSelChip();

end
endtask
